/**
 * Aggressor Adventures Document Library JavaScript
 * Handles search functionality, category toggling, and PDF viewing
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeLibrary();
});

/**
 * Initialize the document library functionality
 */
function initializeLibrary() {
    setupSearch();
    setupCategoryToggles();
    setupKeyboardNavigation();
    setupTouchGestures();
}

/**
 * Setup search functionality
 */
function setupSearch() {
    const searchInput = document.getElementById('searchInput');
    const searchButton = document.querySelector('.search-button');
    
    if (!searchInput) return;
    
    let searchTimeout;
    
    // Real-time search with debouncing
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            performSearch(this.value.trim());
        }, 300);
    });
    
    // Search button click
    if (searchButton) {
        searchButton.addEventListener('click', function() {
            performSearch(searchInput.value.trim());
        });
    }
    
    // Enter key search
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            performSearch(this.value.trim());
        }
    });
}

/**
 * Perform search across all documents
 */
function performSearch(query) {
    const categories = document.querySelectorAll('.category');
    const documentCards = document.querySelectorAll('.document-card');
    
    if (!query) {
        // Show all documents and categories
        categories.forEach(category => {
            category.style.display = 'block';
            const documentsGrid = category.querySelector('.documents-grid');
            if (documentsGrid) {
                documentsGrid.style.display = 'grid';
            }
        });
        
        documentCards.forEach(card => {
            card.style.display = 'block';
            removeHighlight(card);
        });
        
        updateCategoryCounts();
        return;
    }
    
    const searchTerms = query.toLowerCase().split(' ').filter(term => term.length > 0);
    let totalMatches = 0;
    
    categories.forEach(category => {
        const categoryName = category.dataset.category.toLowerCase();
        const documents = category.querySelectorAll('.document-card');
        let categoryMatches = 0;
        
        documents.forEach(card => {
            const filename = card.dataset.filename.toLowerCase();
            const description = card.dataset.description.toLowerCase();
            const searchText = `${filename} ${description} ${categoryName}`;
            
            const matches = searchTerms.every(term => searchText.includes(term));
            
            if (matches) {
                card.style.display = 'block';
                highlightSearchTerms(card, searchTerms);
                categoryMatches++;
                totalMatches++;
            } else {
                card.style.display = 'none';
                removeHighlight(card);
            }
        });
        
        // Show/hide category based on matches
        if (categoryMatches > 0) {
            category.style.display = 'block';
            // Expand category if it has matches
            category.classList.remove('collapsed');
            updateCategoryCount(category, categoryMatches);
        } else {
            category.style.display = 'none';
        }
    });
    
    // Show no results message if needed
    showNoResultsMessage(totalMatches === 0, query);
}

/**
 * Highlight search terms in document cards
 */
function highlightSearchTerms(card, searchTerms) {
    const title = card.querySelector('.document-title');
    const description = card.querySelector('.document-description');
    
    if (title) {
        highlightText(title, searchTerms);
    }
    
    if (description) {
        highlightText(description, searchTerms);
    }
}

/**
 * Highlight text with search terms
 */
function highlightText(element, searchTerms) {
    const originalText = element.textContent;
    let highlightedText = originalText;
    
    searchTerms.forEach(term => {
        const regex = new RegExp(`(${escapeRegex(term)})`, 'gi');
        highlightedText = highlightedText.replace(regex, '<mark>$1</mark>');
    });
    
    if (highlightedText !== originalText) {
        element.innerHTML = highlightedText;
    }
}

/**
 * Remove highlights from document card
 */
function removeHighlight(card) {
    const highlightedElements = card.querySelectorAll('mark');
    highlightedElements.forEach(element => {
        const parent = element.parentNode;
        parent.replaceChild(document.createTextNode(element.textContent), element);
        parent.normalize();
    });
}

/**
 * Escape regex special characters
 */
function escapeRegex(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

/**
 * Update category document count
 */
function updateCategoryCount(category, count) {
    const countElement = category.querySelector('.document-count');
    if (countElement) {
        countElement.textContent = `(${count})`;
    }
}

/**
 * Update all category counts
 */
function updateCategoryCounts() {
    const categories = document.querySelectorAll('.category');
    
    categories.forEach(category => {
        const visibleDocuments = category.querySelectorAll('.document-card[style*="block"], .document-card:not([style])');
        updateCategoryCount(category, visibleDocuments.length);
    });
}

/**
 * Show/hide no results message
 */
function showNoResultsMessage(show, query) {
    let noResultsDiv = document.getElementById('noResults');
    
    if (show) {
        if (!noResultsDiv) {
            noResultsDiv = document.createElement('div');
            noResultsDiv.id = 'noResults';
            noResultsDiv.className = 'empty-state';
            noResultsDiv.innerHTML = `
                <h2>No Documents Found</h2>
                <p>No documents match your search for "<strong>${escapeHtml(query)}</strong>".</p>
                <p>Try different keywords or check the spelling.</p>
            `;
            
            const librarySection = document.querySelector('.library-section');
            librarySection.appendChild(noResultsDiv);
        }
    } else {
        if (noResultsDiv) {
            noResultsDiv.remove();
        }
    }
}

/**
 * Escape HTML special characters
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Setup category toggle functionality
 */
function setupCategoryToggles() {
    const categoryHeaders = document.querySelectorAll('.category-header');
    
    categoryHeaders.forEach(header => {
        header.addEventListener('click', function() {
            const category = this.closest('.category');
            toggleCategory(category);
        });
    });
}

/**
 * Toggle category expanded/collapsed state
 */
function toggleCategory(category) {
    category.classList.toggle('collapsed');
    
    const toggle = category.querySelector('.category-toggle');
    const isCollapsed = category.classList.contains('collapsed');
    
    if (toggle) {
        toggle.setAttribute('aria-expanded', !isCollapsed);
    }
    
    // Save state to localStorage
    const categoryName = category.dataset.category;
    if (categoryName) {
        localStorage.setItem(`category_${categoryName}`, isCollapsed ? 'collapsed' : 'expanded');
    }
}

/**
 * Setup keyboard navigation
 */
function setupKeyboardNavigation() {
    document.addEventListener('keydown', function(e) {
        // Escape key closes PDF modal
        if (e.key === 'Escape') {
            closePDFViewer();
        }
        
        // Ctrl/Cmd + F focuses search
        if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
            e.preventDefault();
            const searchInput = document.getElementById('searchInput');
            if (searchInput) {
                searchInput.focus();
                searchInput.select();
            }
        }
    });
}

/**
 * Setup touch gestures for mobile
 */
function setupTouchGestures() {
    // Add touch-friendly hover effects
    const documentCards = document.querySelectorAll('.document-card');
    
    documentCards.forEach(card => {
        card.addEventListener('touchstart', function() {
            this.classList.add('touch-active');
        });
        
        card.addEventListener('touchend', function() {
            setTimeout(() => {
                this.classList.remove('touch-active');
            }, 150);
        });
    });
}

/**
 * View PDF document in modal
 */
function viewDocument(relativePath, title) {
    const modal = document.getElementById('pdfModal');
    const pdfViewer = document.getElementById('pdfViewer');
    const pdfTitle = document.getElementById('pdfTitle');
    
    if (!modal || !pdfViewer || !pdfTitle) {
        // Fallback: open in new tab
        window.open(`?pdf=${encodeURIComponent(relativePath)}`, '_blank');
        return;
    }
    
    // Set PDF source and title
    pdfViewer.src = `?pdf=${encodeURIComponent(relativePath)}`;
    pdfTitle.textContent = title;
    
    // Show modal
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
    
    // Focus management for accessibility
    const closeButton = modal.querySelector('.modal-close');
    if (closeButton) {
        closeButton.focus();
    }
}

/**
 * Close PDF viewer modal
 */
function closePDFViewer() {
    const modal = document.getElementById('pdfModal');
    const pdfViewer = document.getElementById('pdfViewer');
    
    if (modal) {
        modal.classList.remove('show');
        document.body.style.overflow = '';
        
        // Clear PDF source to stop loading
        if (pdfViewer) {
            pdfViewer.src = '';
        }
    }
}

/**
 * Handle modal click outside to close
 */
document.addEventListener('click', function(e) {
    const modal = document.getElementById('pdfModal');
    if (modal && e.target === modal) {
        closePDFViewer();
    }
});

/**
 * Restore category states from localStorage
 */
function restoreCategoryStates() {
    const categories = document.querySelectorAll('.category');
    
    categories.forEach(category => {
        const categoryName = category.dataset.category;
        if (categoryName) {
            const savedState = localStorage.getItem(`category_${categoryName}`);
            if (savedState === 'collapsed') {
                category.classList.add('collapsed');
                const toggle = category.querySelector('.category-toggle');
                if (toggle) {
                    toggle.setAttribute('aria-expanded', 'false');
                }
            }
        }
    });
}

// Restore category states when page loads
document.addEventListener('DOMContentLoaded', restoreCategoryStates);

/**
 * Handle window resize for responsive behavior
 */
window.addEventListener('resize', function() {
    // Adjust modal size on orientation change
    const modal = document.getElementById('pdfModal');
    if (modal && modal.classList.contains('show')) {
        // Small delay to ensure orientation change is complete
        setTimeout(() => {
            const pdfViewer = document.getElementById('pdfViewer');
            if (pdfViewer) {
                // Force iframe to recalculate size
                const src = pdfViewer.src;
                pdfViewer.src = '';
                setTimeout(() => {
                    pdfViewer.src = src;
                }, 100);
            }
        }, 300);
    }
});
