<?php
/**
 * Aggressor Adventures Document Library - MAMP Compatible Version
 * Simplified version for better MAMP compatibility
 */

// Error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configuration
$DOCS_DIR = dirname(__FILE__) . '/docs';
$ALLOWED_EXTENSIONS = array('pdf');
$METADATA_EXTENSIONS = array('json', 'txt');

// Simple function to scan documents
function scanDocuments($baseDir, $allowedExt, $metaExt) {
    $library = array();
    
    if (!is_dir($baseDir)) {
        return $library;
    }
    
    $categories = scandir($baseDir);
    
    foreach ($categories as $category) {
        if ($category === '.' || $category === '..') {
            continue;
        }
        
        $categoryPath = $baseDir . '/' . $category;
        
        if (!is_dir($categoryPath)) {
            continue;
        }
        
        $documents = array();
        $files = scandir($categoryPath);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $filePath = $categoryPath . '/' . $file;
            $fileInfo = pathinfo($file);
            
            if (!isset($fileInfo['extension']) || 
                !in_array(strtolower($fileInfo['extension']), $allowedExt)) {
                continue;
            }
            
            $description = '';
            // Try to get description from metadata files
            foreach ($metaExt as $ext) {
                $metadataFile = $categoryPath . '/' . $fileInfo['filename'] . '.' . $ext;
                if (file_exists($metadataFile)) {
                    $content = file_get_contents($metadataFile);
                    if ($ext === 'json') {
                        $data = json_decode($content, true);
                        if (isset($data['description'])) {
                            $description = $data['description'];
                        }
                    } else {
                        $description = trim($content);
                    }
                    break;
                }
            }
            
            $document = array(
                'filename' => $fileInfo['filename'],
                'file' => $file,
                'path' => $filePath,
                'relative_path' => $category . '/' . $file,
                'size' => filesize($filePath),
                'modified' => filemtime($filePath),
                'description' => $description
            );
            
            $documents[] = $document;
        }
        
        if (!empty($documents)) {
            $library[$category] = $documents;
        }
    }
    
    return $library;
}

// Handle PDF serving
if (isset($_GET['pdf']) && !empty($_GET['pdf'])) {
    $requestedPath = $_GET['pdf'];
    $pdfPath = $DOCS_DIR . '/' . $requestedPath;

    // Debug information (remove in production)
    if (isset($_GET['debug'])) {
        echo "<h3>Debug Information</h3>";
        echo "Requested path: " . htmlspecialchars($requestedPath) . "<br>";
        echo "Full PDF path: " . htmlspecialchars($pdfPath) . "<br>";
        echo "DOCS_DIR: " . htmlspecialchars($DOCS_DIR) . "<br>";
        echo "File exists: " . (file_exists($pdfPath) ? 'Yes' : 'No') . "<br>";
        echo "File readable: " . (is_readable($pdfPath) ? 'Yes' : 'No') . "<br>";
        echo "Real path: " . htmlspecialchars(realpath($pdfPath)) . "<br>";
        exit;
    }

    // Security check
    $realDocsDir = realpath($DOCS_DIR);
    $realPdfPath = realpath($pdfPath);

    if ($realPdfPath && $realDocsDir && strpos($realPdfPath, $realDocsDir) === 0) {
        if (file_exists($realPdfPath) && is_readable($realPdfPath)) {
            $fileInfo = pathinfo($realPdfPath);
            if (isset($fileInfo['extension']) && strtolower($fileInfo['extension']) === 'pdf') {
                header('Content-Type: application/pdf');
                header('Content-Length: ' . filesize($realPdfPath));
                header('Content-Disposition: inline; filename="' . basename($realPdfPath) . '"');
                readfile($realPdfPath);
                exit;
            }
        }
    }

    // Enhanced error message for debugging
    http_response_code(404);
    echo "<h3>File Not Found</h3>";
    echo "Requested: " . htmlspecialchars($requestedPath) . "<br>";
    echo "Looking for: " . htmlspecialchars($pdfPath) . "<br>";
    echo "File exists: " . (file_exists($pdfPath) ? 'Yes' : 'No') . "<br>";
    echo "<a href='?debug=1&pdf=" . urlencode($requestedPath) . "'>Debug this request</a>";
    exit;
}

// Scan document library
$documentLibrary = scanDocuments($DOCS_DIR, $ALLOWED_EXTENSIONS, $METADATA_EXTENSIONS);

// Simple file size formatting
function formatBytes($size) {
    $units = array('B', 'KB', 'MB', 'GB');
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    return round($size, 1) . ' ' . $units[$i];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aggressor Adventures - Document Library</title>
    <style>
        /* Inline CSS for MAMP compatibility */
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #2d3748;
            background-color: #f7fafc;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        .header {
            background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%);
            color: white;
            padding: 1rem 0;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .header h1 {
            font-size: 1.8rem;
            margin-bottom: 0.25rem;
        }
        
        .tagline {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .main {
            padding: 2rem 0;
        }
        
        .search-section {
            margin-bottom: 2rem;
            text-align: center;
        }
        
        .search-input {
            width: 100%;
            max-width: 500px;
            padding: 1rem;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 1rem;
        }
        
        .category {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
            overflow: hidden;
        }
        
        .category-header {
            background: linear-gradient(135deg, #e6f3ff 0%, white 100%);
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            cursor: pointer;
        }
        
        .category-title {
            font-size: 1.4rem;
            color: #1e3a5f;
            margin: 0;
        }
        
        .documents-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.5rem;
            padding: 1.5rem;
        }
        
        .document-card {
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .document-card:hover {
            border-color: #2c5282;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            transform: translateY(-2px);
        }
        
        .document-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: #1e3a5f;
            margin-bottom: 0.5rem;
        }
        
        .document-description {
            color: #718096;
            font-size: 0.9rem;
            margin-bottom: 1rem;
        }
        
        .document-meta {
            display: flex;
            justify-content: space-between;
            font-size: 0.8rem;
            color: #718096;
            margin-bottom: 1rem;
        }
        
        .document-view-btn {
            width: 100%;
            background: linear-gradient(135deg, #38a169 0%, #2f855a 100%);
            color: white;
            border: none;
            padding: 0.75rem;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
        }
        
        .document-view-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .footer {
            background-color: #1e3a5f;
            color: white;
            text-align: center;
            padding: 1.5rem 0;
            margin-top: 3rem;
        }
        
        @media (max-width: 768px) {
            .documents-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
                padding: 1rem;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="container">
            <h1>Aggressor Adventures</h1>
            <span class="tagline">Document Library</span>
        </div>
    </header>

    <main class="main">
        <div class="container">
            <div class="search-section">
                <input type="text" id="searchInput" class="search-input" placeholder="Search documents..." onkeyup="searchDocuments()">
            </div>

            <div class="library-section">
                <?php if (empty($documentLibrary)): ?>
                    <div class="empty-state">
                        <h2>No Documents Found</h2>
                        <p>Please add PDF documents to the docs/ directory to get started.</p>
                        <p><strong>Debug Info:</strong></p>
                        <p>Docs directory: <?php echo htmlspecialchars($DOCS_DIR); ?></p>
                        <p>Directory exists: <?php echo is_dir($DOCS_DIR) ? 'Yes' : 'No'; ?></p>
                        <p>Directory readable: <?php echo is_readable($DOCS_DIR) ? 'Yes' : 'No'; ?></p>
                        <?php
                        // Show what files are actually in the docs directory
                        if (is_dir($DOCS_DIR)) {
                            echo "<p><strong>Directory contents:</strong></p><ul>";
                            $items = scandir($DOCS_DIR);
                            foreach ($items as $item) {
                                if ($item !== '.' && $item !== '..') {
                                    $itemPath = $DOCS_DIR . '/' . $item;
                                    echo "<li>" . htmlspecialchars($item) . " (" . (is_dir($itemPath) ? 'directory' : 'file') . ")";
                                    if (is_dir($itemPath)) {
                                        $subItems = scandir($itemPath);
                                        echo "<ul>";
                                        foreach ($subItems as $subItem) {
                                            if ($subItem !== '.' && $subItem !== '..') {
                                                echo "<li>" . htmlspecialchars($subItem) . "</li>";
                                            }
                                        }
                                        echo "</ul>";
                                    }
                                    echo "</li>";
                                }
                            }
                            echo "</ul>";
                        }
                        ?>
                    </div>
                <?php else: ?>
                    <?php foreach ($documentLibrary as $categoryName => $documents): ?>
                        <div class="category">
                            <div class="category-header" onclick="toggleCategory(this)">
                                <h2 class="category-title">
                                    📁 <?php echo htmlspecialchars($categoryName); ?>
                                    (<?php echo count($documents); ?>)
                                </h2>
                            </div>
                            
                            <div class="documents-grid">
                                <?php foreach ($documents as $document): ?>
                                    <div class="document-card" onclick="viewDocument('<?php echo htmlspecialchars($document['relative_path']); ?>', '<?php echo htmlspecialchars($document['filename']); ?>')">
                                        <div style="font-size: 2rem; text-align: center; margin-bottom: 1rem;">📄</div>
                                        <h3 class="document-title"><?php echo htmlspecialchars($document['filename']); ?></h3>
                                        <?php if (!empty($document['description'])): ?>
                                            <p class="document-description"><?php echo htmlspecialchars($document['description']); ?></p>
                                        <?php endif; ?>
                                        <div class="document-meta">
                                            <span><?php echo formatBytes($document['size']); ?></span>
                                            <span><?php echo date('M j, Y', $document['modified']); ?></span>
                                        </div>
                                        <button class="document-view-btn">View PDF</button>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Aggressor Adventures. All rights reserved.</p>
        </div>
    </footer>

    <script>
        function searchDocuments() {
            var input = document.getElementById('searchInput');
            var filter = input.value.toLowerCase();
            var cards = document.getElementsByClassName('document-card');
            
            for (var i = 0; i < cards.length; i++) {
                var title = cards[i].getElementsByClassName('document-title')[0];
                var description = cards[i].getElementsByClassName('document-description')[0];
                
                var titleText = title ? title.textContent.toLowerCase() : '';
                var descText = description ? description.textContent.toLowerCase() : '';
                
                if (titleText.indexOf(filter) > -1 || descText.indexOf(filter) > -1) {
                    cards[i].style.display = '';
                } else {
                    cards[i].style.display = 'none';
                }
            }
        }
        
        function toggleCategory(header) {
            var grid = header.nextElementSibling;
            if (grid.style.display === 'none') {
                grid.style.display = 'grid';
            } else {
                grid.style.display = 'none';
            }
        }
        
        function viewDocument(path, title) {
            window.open('?pdf=' + encodeURIComponent(path), '_blank');
        }
    </script>
</body>
</html>
